<?php
/**
 * Utilisations de pipelines par Biodiv
 *
 * @plugin     Biodiv
 * @copyright  2020
 * @author     Renaud LAURETTE
 * @licence    GNU/GPL
 * @package    SPIP\Biodiv\Pipelines
 */

if (!defined('_ECRIRE_INC_VERSION')) {
	return;
}


/*
 * Un fichier de pipelines permet de regrouper
 * les fonctions de branchement de votre plugin
 * sur des pipelines existants.
 */


/**
 * Ajouter les objets sur les vues des parents directs
 *
 * @pipeline affiche_enfants
 * @param  array $flux Données du pipeline
 * @return array       Données du pipeline
**/
function biodiv_affiche_enfants($flux) {
	if ($e = trouver_objet_exec($flux['args']['exec']) and $e['edition'] == false) {
		$id_objet = $flux['args']['id_objet'];

		if ($e['type'] == 'rubrique') {

			$flux['data'] .= recuperer_fond(
				'prive/objets/liste/biodiv_campagnes',
				array(
					'titre' => _T('campagne:titre_biodiv_campagnes_rubrique'),
					'id_rubrique' => $id_objet
				)
			);

			if (autoriser('creercampagnedans', 'rubrique', $id_objet)) {
				include_spip('inc/presentation');
				$flux['data'] .= icone_verticale(
					_T('campagne:icone_creer_campagne'),
					generer_url_ecrire('campagne_edit', "id_rubrique=$id_objet"),
					'campagne-24.png',
					'new',
					'right'
				) . "<br class='nettoyeur' />";
			}

		}
	}
	return $flux;
}


/**
 * Ajout de contenu sur certaines pages,
 * notamment des formulaires de liaisons entre objets
 *
 * @pipeline affiche_milieu
 * @param  array $flux Données du pipeline
 * @return array       Données du pipeline
 */
function biodiv_affiche_milieu($flux) {
	$texte = '';
	$e = trouver_objet_exec($flux['args']['exec']);

	// auteurs sur les biodiv_observations, biodiv_sources, biodiv_campagnes
	if (($e != false) and !$e['edition'] and in_array($e['type'], array('observation', 'source', 'campagne'))) {
		$texte .= recuperer_fond('prive/objets/editer/liens', array(
			'table_source' => 'auteurs',
			'objet' => $e['type'],
			'id_objet' => $flux['args'][$e['id_table_objet']]
		));
	}


	// biodiv_observations sur les articles
    // Imaginé pour relier des observations à des articles 'expo' mais
	// retiré car ambigu: pourrait (mais ne doit pas) être interprété comme le
	// lien entre observation et fiche espèce.
	/*
	if (!$e['edition'] and in_array($e['type'], array('article'))) {
		$texte .= recuperer_fond('prive/objets/editer/liens', array(
			'table_source' => 'biodiv_observations',
			'objet' => $e['type'],
			'id_objet' => $flux['args'][$e['id_table_objet']]
		));
	}
	*/

	if ($texte) {
		if ($p = strpos($flux['data'], '<!--affiche_milieu-->')) {
			$flux['data'] = substr_replace($flux['data'], $texte, $p, 0);
		} else {
			$flux['data'] .= $texte;
		}
	}

	return $flux;
}


/**
 * Ajout de liste sur la vue d'un auteur
 *
 * @pipeline affiche_auteurs_interventions
 * @param  array $flux Données du pipeline
 * @return array       Données du pipeline
 */
function biodiv_affiche_auteurs_interventions($flux) {
	if ($id_auteur = intval($flux['args']['id_auteur'])) {
		$flux['data'] .= recuperer_fond('prive/objets/liste/biodiv_observations', array(
			'id_auteur' => $id_auteur,
			'titre' => _T('observation:info_biodiv_observations_auteur')
		), array('ajax' => true));
	
		$flux['data'] .= recuperer_fond('prive/objets/liste/biodiv_sources', array(
			'id_auteur' => $id_auteur,
			'titre' => _T('source:info_biodiv_sources_auteur')
		), array('ajax' => true));
	
		$flux['data'] .= recuperer_fond('prive/objets/liste/biodiv_campagnes', array(
			'id_auteur' => $id_auteur,
			'titre' => _T('campagne:info_biodiv_campagnes_auteur')
		), array('ajax' => true));
	}
	return $flux;
}

/**
 * Afficher le nombre d'éléments dans les parents
 *
 * @pipeline boite_infos
 * @param  array $flux Données du pipeline
 * @return array       Données du pipeline
**/
function biodiv_boite_infos($flux) {
	if (isset($flux['args']['type']) and isset($flux['args']['id']) and $id = intval($flux['args']['id'])) {
		$texte = '';
		if ($flux['args']['type'] == 'rubrique' and $nb = sql_countsel('spip_biodiv_campagnes', array("statut='publie'", 'id_rubrique=' . $id))) {
			$texte .= '<div>' . singulier_ou_pluriel($nb, 'campagne:info_1_campagne', 'campagne:info_nb_biodiv_campagnes') . "</div>\n";
		}
		if ($texte and $p = strpos($flux['data'], '<!--nb_elements-->')) {
			$flux['data'] = substr_replace($flux['data'], $texte, $p, 0);
		}
	}
	return $flux;
}


/**
 * Compter les enfants d'un objet
 *
 * @pipeline objets_compte_enfants
 * @param  array $flux Données du pipeline
 * @return array       Données du pipeline
**/
function biodiv_objet_compte_enfants($flux) {
	if ($flux['args']['objet'] == 'rubrique' and $id_rubrique = intval($flux['args']['id_objet'])) {
		// juste les publiés ?
		if (array_key_exists('statut', $flux['args']) and ($flux['args']['statut'] == 'publie')) {
			$flux['data']['biodiv_campagnes'] = sql_countsel('spip_biodiv_campagnes', 'id_rubrique= ' . intval($id_rubrique) . " AND (statut = 'publie')");
		} else {
			$flux['data']['biodiv_campagnes'] = sql_countsel('spip_biodiv_campagnes', 'id_rubrique= ' . intval($id_rubrique) . " AND (statut <> 'poubelle')");
		}
	}

	return $flux;
}


/**
 * Optimiser la base de données
 *
 * Supprime les liens orphelins de l'objet vers quelqu'un et de quelqu'un vers l'objet.
 * Supprime les objets à la poubelle.
 *
 * @pipeline optimiser_base_disparus
 * @param  array $flux Données du pipeline
 * @return array       Données du pipeline
 */
function biodiv_optimiser_base_disparus($flux) {

	include_spip('action/editer_liens');
	$flux['data'] += objet_optimiser_liens(array('observation'=>'*'), '*');

	sql_delete('spip_biodiv_campagnes', "statut='poubelle' AND maj < " . sql_quote($flux['args']['date']));

	return $flux;
}

/**
 * Synchroniser la valeur de id secteur
 *
 * @pipeline trig_propager_les_secteurs
 * @param  string $flux Données du pipeline
 * @return string       Données du pipeline
**/
function biodiv_trig_propager_les_secteurs($flux) {

	// synchroniser spip_biodiv_campagnes
	$r = sql_select(
		'A.id_campagne AS id, R.id_secteur AS secteur',
		'spip_biodiv_campagnes AS A, spip_rubriques AS R',
		'A.id_rubrique = R.id_rubrique AND A.id_secteur <> R.id_secteur'
	);
	while ($row = sql_fetch($r)) {
		sql_update('spip_biodiv_campagnes', array('id_secteur' => $row['secteur']), 'id_campagne=' . $row['id']);
	}

	return $flux;
}



/**
 * Ajouts hors Fabrique
 * --------------------
 */

 /**
 * Taches périodiques
 *
 * Donne la périodicité des tâches en secondes.
 * La $tache['nom'] est décite par la fonction genie_nom_dist() dans le
 * fichier genie/nom.php
 *
 * @pipeline taches_generales_cron
 * @param  array $taches tableau de tâches à exécuter
 * @return array      	 tableau de tâches à exécuter
**/
function biodiv_taches_generales_cron($taches){
	
	// $taches['taxref'] = 10; 	// Indexation TAXREF	
	$taches['taxref_v2'] = 10; 	// Indexation TAXREF V2	
	$taches['logomig'] = 30;	// Migration des logos
	
	// $taches['sensible'] = 3600; // tous les heures
	spip_log("CRON - taches enregitreees : " . print_r($taches,true), "biodiv");
	return $taches;
}

/**
 * Ajout d'information sur des objets dans l'espace privé
 *
 * Teste le type d'objet $args["args"]["type"]
 * et ajoute à $args["data"] un fragment de squelette
 * qui sera typiquement dans "prive/objets/extra/nom-objet"
 * en passant à recuperer_fond() des éléments de contexte.
 *
 * @pipeline afficher_complement_objet
 * @param  array $args information sur l'objet courant
 * @return array       le même tableau enrichi
**/
function biodiv_afficher_complement_objet($args) {
	// Les observations
	/*
	if ($args["args"]["type"] == "observation") {
		$args["data"] .= recuperer_fond("prive/objets/extra/observation",
			array('id_observation' => $args["args"]["id"]));
	}
	*/
	// autres
	
	// Retour
	return $args;
}
 
function biodiv_header_prive($flux)
{
	$flux .= "\n<!-- biodiv prive start -->\n";
	/*
	$flux .="\n".'<script src="'.url_absolue(find_in_path(_DIR_LIB_LEAFLET.'leaflet.js')).'"></script>'."\n";
	$flux .="\n".'<link rel="stylesheet" href="'.url_absolue(find_in_path(_DIR_LIB_LEAFLET.'leaflet.css')).'" />'."\n";
	$flux .="\n".'<link rel="stylesheet" href="'.url_absolue(find_in_path(_DIR_LIB.'leaflet-fullscreen/Control.FullScreen.css')).'" />'."\n";
	*/
	$flux .="\n".'<link rel="stylesheet" href="'.url_absolue(find_in_path('obsform_private.css')).'" />'."\n";
	$flux .= "<!-- biodiv prive end -->\n";

	return $flux;
}

// Pipeline. Pour l'entete des pages de l'espace public
function biodiv_insert_head($flux)
{	
	$flux .= "<!-- biodiv start -->\n";
	$flux .="\n".'<script src="'.url_absolue(find_in_path(_DIR_PLUGIN_BIODIV.'js/feuillet_biodiv.js')).'"></script>'."\n";
	/*
	$flux .="\n".'<link rel="stylesheet" href="'.url_absolue(find_in_path(_DIR_LIB_LEAFLET.'leaflet.css')).'" />'."\n";
	$flux .="\n".'<link rel="stylesheet" href="'.url_absolue(find_in_path(_DIR_LIB.'leaflet-fullscreen/Control.FullScreen.css')).'" />'."\n";
	*/
	$flux .= "<!-- biodiv end -->\n";	
	return $flux;
}

/**
 * Traitement post-edition des objets
 *
 * Intercepte le changement des statuts des objets afin de purger les liens
 * entre les observations et les objets dépubliés.
 *
 * @pipeline post_edition
 * @param  array $flux les données entrantes
 * @return array       les mêmes en sortie
**/

function biodiv_post_edition($flux) {
	
	// Interception du changement de statut
	if($flux['args']['action'] == 'instituer') {
        $objet = objet_type($flux['args']['table']);
        if (isset($flux['data']['statut']) && ($flux['data']['statut'] != 'publie'))	{
        	// On controle les références aux objets disparus
        	$ido = $flux['args']['id_objet'];
        	$statut = $flux['data']['statut'];
        	switch($objet) {
        		case 'rubrique': 
        			decrocher_famille($ido,$statut); 
        			break;
        		case 'article':  
        			decrocher_espece($ido, $statut); 
        			break;	
        		default: break;
        	}
        }
    }
    return $flux;  			
}

/**
 * Traitement post-insertion
 * 
 * @pipeline post_insertion
 * @param array données en entrée
 * @return array	données inchangées
**//*
function biodiv_post_insertion($flux) {
	
	spip_log("BIODIV::PostInsertion ".serialize($flux),"biodiv."._LOG_INFO_IMPORTANTE);
	if($flux['args']['table'] == 'spip_biodiv_observations') {
		spip_log("BIODIV::PostInsertion observation","biodiv."._LOG_INFO_IMPORTANTE);
		// insertion d'une observation
		if(test_plugin_actif('sibmail')) {
			// Quand sibmail est actif, on envoie un évenement "creationObservation"
			spip_log("BIODIV::PostInsertion observation - sibmail actif","biodiv."._LOG_INFO_IMPORTANTE);
			$data = new stdClass(); 
			$data->titre = $flux['data']['titre'];
			$data->date_obs = $flux['data']['date_obs'];
			$data->type_obs = $flux['data']['type_obs'];
			$data->espece = $flux['data']['espece'];
			$data->quantite = $flux['data']['quantite'];
			
			$props = new stdClass();
			$pseudo = lire_config('sibmail/attribut_pseudo');
			$props->$pseudo = $GLOBALS['visiteur_session']['nom'];
			
			$st = sibmail_sendEvent(
				$GLOBALS['visiteur_session']['email'],
				'creerObservation',
				$flux['args']['id_objet'],
				$props, $data);
			spip_log("BIODIV::PostInsertion observation: $st","biodiv."._LOG_INFO_IMPORTANTE);
		}
    }

	return $flux;
}*/

/**
 * Interception des traitements de formulaires pour envoyer des evenements Brevo
 *
 * @param array $flux	Flux de données du formulaire 
 * @return array		Le flux inchangé
 *
*/

function biodiv_formulaire_traiter($flux) {
	
	$observation = null;
	
	// FORMULAIRE_EDITER_OBSERVATION - Création via formulaire standard
	if($flux['args']['form'] == 'editer_observation') {
		$id = intval((string)$flux['args']['args'][0]);
		if($id == 0) {
			// L'identifiant n'est pas un entier : c'est une création
			$observation = $flux['data']['id_observation'];
			spip_log("BIODIV::FormulaireTraiter création obs: ".$flux['data']['id_observation'],"biodiv."._LOG_INFO_IMPORTANTE);
		} else {
			spip_log("BIODIV::FormulaireTraiter modification obs: ".$flux['data']['id_observation'],"biodiv."._LOG_INFO_IMPORTANTE);
		}
	} 
	
	// FORMULAIRE_SAISIR_OBSERVATION - Saisie immédiate
	if($flux['args']['form'] == 'saisir_observation') {
		$observation = $flux['data']['id_observation'];
		spip_log("BIODIV::FormulaireTraiter saisie obs: ".$flux['data']['id_observation'],"biodiv."._LOG_INFO_IMPORTANTE);
	}
	
	// FORMULAIRE_ENREGISTRER_OBSERVATIONS - Saisie depuis parcours GPS
	if($flux['args']['form'] == 'enregistrer_observations') {
		spip_log("BIODIV::FormulaireTraiter enregistrer obs: data= ".serialize($flux['data']),"biodiv."._LOG_INFO_IMPORTANTE);
	}	
	
	// FORMULAIRE_ORDONNER_BIODIV - Import freemind
	if($flux['args']['form'] == 'ordonner_biodiv') {
		spip_log("BIODIV::FormulaireTraiter ordonner biodiv: data= ".serialize($flux['data']),"biodiv."._LOG_INFO_IMPORTANTE);
	}
	
	// FORMULAIRE_IMPORTER_BIODIV - Import XML
	if($flux['args']['form'] == 'importer_biodiv') {
		spip_log("BIODIV::FormulaireTraiter importer biodiv: data= ".serialize($flux['data']),"biodiv."._LOG_INFO_IMPORTANTE);
	}
		
	// FORMULAIRE_FORUM - commentaire sur un objet
	if($flux['args']['form'] == 'forum') {		
		// spip_log("BIODIV::FormulaireTraiter forum: data= ".serialize($flux['data']),"biodiv."._LOG_INFO_IMPORTANTE);
		if(test_plugin_actif('sibmail')) notifier_forum_sibmail($flux);
	}
	
	// Evenements de création d'observation
	if($observation and test_plugin_actif('sibmail')) {
		$row = sql_fetsel("*",'spip_biodiv_observations','id_observation='.sql_quote($observation));
		$data = new stdClass(); 
		$data->id_observation = $observation;
		$data->titre = $row['titre'];
		$data->date_obs = $row['date_obs'];
		$data->type_obs = $row['type_obs'];
		$data->espece = $row['espece'];
		$data->quantite = $row['quantite'];
		$data->url_observation = generer_url_public('observation','id_observation='.$observation,true,false);
		
			
		$props = new stdClass();
		$pseudo = strtoupper(lire_config('sibmail/attribut_pseudo'));
		$statutSite = strtoupper(lire_config('sibmail/attribut_statut'));
		$props->$pseudo = $GLOBALS['visiteur_session']['nom'];
		$props->$statutSite = _T('sibmail:'.$GLOBALS['visiteur_session']['statut']);	
		
		$st = sibmail_sendEvent(
			$GLOBALS['visiteur_session']['email'],
			'creerObservation',
			(string)$observation,
			$props, $data);
		spip_log("BIODIV::PostInsertion observation: $st","biodiv."._LOG_INFO_IMPORTANTE);
	}
	
	return $flux;
}


/**
 * Interception des traitements de formulaires pour envoyer des evenements Brevo
 * Traitement spécifique des formulaires de forum
 *
 * @param array $flux	Flux de données du formulaire 
 * @return array		Le flux inchangé
 *
*/

function notifier_forum_sibmail($flux) {

	$forum = sql_fetsel("*",'spip_forum', 'id_forum='.sql_quote($flux['data']['id_forum']));
	$pseudo = lire_config('sibmail/attribut_pseudo');
	$eventid = $forum['objet'] . $forum['id_objet'] . ':forum';
	
	// On construit une liste des intervenants pour ne pas notifier deux fois la même personne
	$intervenants = array($forum['email_auteur']);
	
	$props = new stdClass();
	
	$data = new stdClass();
	$data->operateur = $forum['auteur'];
	$data->objet = $forum['objet'];
	$data->id_objet = $forum['id_objet'];
	
	// Recherche des informations de l'objet
	$table_sql = table_objet_sql($forum['objet']);
	$trouver_table = charger_fonction('trouver_table', 'base');
	$desc = $trouver_table($table_sql);
	if ($desc and isset($desc['field']) and isset($desc['titre']) and $forum['id_objet']) {
		// on cherche le titre de l'objet
		$titre = strtok($desc['titre']," ,\t\n");
		$pk = $desc['key']['PRIMARY KEY'];
		$row = sql_fetsel($titre, $table_sql, $pk . "=" . sql_quote($forum['id_objet']));
		$data->titre_objet = $row[$titre];
		$data->url_objet = generer_url_public($forum['objet'],$pk.'='.$forum['id_objet'],true,false);
	}
	// Recherche des auteurs de l'objet
	$res = sql_select(
		array( // select
			"a.id_auteur AS id_auteur",
			"a.nom AS nom",
			"a.email AS email"
			),
		array( // from
			"spip_auteurs AS a", 
			"spip_auteurs_liens AS liens"
			),
		array( // where
			"liens.objet=".sql_quote($forum['objet']), 
			"liens.id_objet=".sql_quote($forum['id_objet']),
			"liens.id_auteur=a.id_auteur"
			)
		);
	
	// On notifie les auteurs de l'objet
	while($r = sql_fetch($res)) {
		$props->$pseudo = $r['nom'];
		if(!in_array($r['email'],$intervenants)) {
			// On evite les envois multiples
			sibmail_sendEvent($r['email'],'commenter',$eventid,$props,$data);
			// spip_log("BIODIV::Forum ".serialize($data)." pour ".serialize($props),"biodiv."._LOG_INFO_IMPORTANTE);
			array_push($intervenants, $r['email']);
		}
	}
	// On recherche les participants à la conversation	et on les notifie
	$thread = sql_select("*",'spip_forum','id_thread='.sql_quote($forum['id_thread']));
	while($r = sql_fetch($thread)) {
		$props->$pseudo = $r['auteur'];
		if(!in_array($r['email_auteur'],$intervenants)) {
			// On evite les envois multiples
			sibmail_sendEvent($r['email_auteur'],'commenter',$eventid,$props,$data);
			// spip_log("BIODIV::Forum ".serialize($data)." pour ".serialize($props),"biodiv."._LOG_INFO_IMPORTANTE);
			array_push($intervenants, $r['email_auteur']);
		}
	}
}


/**
 * Envoi d'événement Brevo lors de la modification d'un lien avec une observation
 *
 * @param object $params	Structure principale de l'evenement
 *
 * 		pipeline('apres_lien_observation',array(
			'id_observation' => $id_observation,
			'objet' => 'rubrique',
			'nature' => 'famille',
			'id_objet' => $id_rubrique
			));	
 *
*/
function biodiv_apres_lien_observation($params) {
	spip_log("BIODIV::ApresLienObservation params= ".print_r($params,true),"biodiv."._LOG_INFO_IMPORTANTE);
	if(test_plugin_actif('sibmail') and (intval($params['id_objet']) > 0)) {
		// On ne génère d'évenement que lors d'un lien avec un vrai objet et quand le plugin est actif
		// Recherche des informations de l'observation
		$obs = sql_fetsel('*','spip_biodiv_observations','id_observation='.$params['id_observation']);
		// Preparation des données
		$data = new stdClass();
		$data->id_observation = $params['id_observation'];
		$data->titre = $obs['titre'];
		$data->date_obs = $obs['date_obs'];
		$data->url_observation = generer_url_public('observation','id_observation='.$params['id_observation'],true,false);
		$data->objet = $params['objet'];
		$data->nature = $params['nature'];
		$data->id_objet = $params['id_objet'];
		// Recherche des informations de l'objet
		$table_sql = table_objet_sql($params['objet']);
		$trouver_table = charger_fonction('trouver_table', 'base');
		$desc = $trouver_table($table_sql);
		if ($desc and isset($desc['field']) and isset($desc['titre']) and $params['id_objet']) {
			// on cherche le titre de l'objet
			$titre = strtok($desc['titre']," ,\t\n");
			$pk = $desc['key']['PRIMARY KEY'];
			$row = sql_fetsel($titre, $table_sql, $pk . "=" . sql_quote($params['id_objet']));
			$data->titre_objet = $row[$titre];
			$data->url_objet = generer_url_public($params['objet'],$pk.'='.$params['id_objet'],true,false);
		}
		
		// Qui a fait le lien ?
		$data->operateur = $GLOBALS['visiteur_session']['nom'];
		$id_operateur = $GLOBALS['visiteur_session']['id_auteur'];
		
		// Quel est l'id de l'événement ?
		$eventid = 'observation'.$params['id_observation'].':'.$params['nature'].$params['id_objet'];

		// On associe un événement à chaque auteur de l'observation
		$props = new stdClass();
		$pseudo = lire_config('sibmail/attribut_pseudo');
		$res = sql_select(
			array( // select
				"a.id_auteur AS id_auteur",
				"a.nom AS nom",
				"a.email AS email"
				),
			array( // from
				"spip_auteurs AS a", 
				"spip_auteurs_liens AS liens"
				),
			array( // where
				"liens.objet='observation'", 
				"liens.id_objet=".$params['id_observation'],
				"liens.id_auteur=a.id_auteur"
				)
			);
		// On compte les observations liées à cet objet
		$nbliens = 0;
		switch($params['nature']) {
		case 'espece':
			$nbliens = sql_countsel("spip_biodiv_observations","id_espece=".sql_quote($params['id_objet']));
			break;
		case 'famille':
			$nbliens = sql_countsel("spip_biodiv_observations","id_famille=".sql_quote($params['id_objet']));
			break;
		case 'sujet':
			$nbliens = sql_countsel("spip_biodiv_observations","id_sujet=".sql_quote($params['id_objet']));
			break;
		case 'campagne':
			$nbliens = sql_countsel("spip_biodiv_observations_liens",array(
						"id_objet=".sql_quote($params['id_objet']),
						"objet='campagne'"
						));
			break;
		case 'auteur':
			$nbliens = sql_countsel("spip-auteurs_liens",array(
						"id_objet=".sql_quote($params['id_observation']),
						"objet='observation'"
						));
		default:
			break;
		}
		$data->nb_liens = $nbliens;
		
		// Envoi d'un événement par observateur
		while($row = sql_fetch($res)) {
			$props->$pseudo = $row['nom'];
			if($row['id_auteur'] != $id_operateur) {
				// On n'envoie l'événement du lien que si l'opérateur n'est pas l'auteur de l'observation
				sibmail_sendEvent($row['email'],'lierObservation',$eventid,$props,$data);
			}
		}
		
	}
	return $params;
}

/**
 * Décrocher les observations des familles (rubriques)
 *
 * Remonte les rubriques jusqu'à trouver une rubrique publiée
 * ou le haut du secteur. Raccroche les observations trouvées
 * à la première rubrique publiée de la hierarchie
 *
 * @pipeline post_edition
 * @param integer $ido 	identifiant de la famille/rubrique
 * @param string $statut   nouveau statut de la rubrique
**/

function decrocher_famille($ido,$statut) {
	 spip_log("POST-EDITION Rubrique $ido ($statut)","biodiv."._LOG_INFO_IMPORTANTE);
	$idp = sql_getfetsel('id_parent','spip_rubriques','id_rubrique='.intval($ido));
	do {
		$idr=$idp;
		$parent = sql_fetsel('*','spip_rubriques','id_rubrique='.intval($idr));
		$idp = $parent['id_parent'];
		$stp = $parent['statut'];
	} while (($idp != 0) and ($stp != 'publie'));
	$resultats = sql_select('*','spip_biodiv_observations','id_famille='.intval($ido));
	$observations = array();
	while($res = sql_fetch($resultats)) {
		$observations[] = $idobs = $res['id_observation'];
		 spip_log("POST-EDITION Rubrique $ido >> Observation $idobs","biodiv."._LOG_INFO_IMPORTANTE);
	}
	if($n=count($observations)) {
		sql_update(
			'spip_biodiv_observations',
			array(
				'id_espece' => 0,
				'id_famille' => intval($idr)
				),
			sql_in('id_observation',$observations)
			);
		spip_log("POST-EDITION Rubrique $ido >> $n Observations déplacées dans rubrique $idr","biodiv."._LOG_INFO_IMPORTANTE);
	}
}

/**
 * Décrocher les observations des espèces (articles)
 *
 * Remonte les rubriques jusqu'à trouver une rubrique publiée
 * ou le haut du secteur. Raccroche les observations trouvées
 * à la première rubrique publiée de la hierarchie
 *
 * @pipeline post_edition
 * @param integer $ido 	identifiant de la famille/rubrique
 * @param string $statut   nouveau statut de la rubrique
**/

function decrocher_espece($ido,$statut) {
	spip_log("POST-EDITION Espèce $ido ($statut)","biodiv."._LOG_INFO_IMPORTANTE);
	$idp = sql_getfetsel('id_rubrique','spip_articles','id_article='.intval($ido));
	do {
		$idr=$idp;
		$parent = sql_fetsel('*','spip_rubriques','id_rubrique='.intval($idr));
		$idp = $parent['id_parent'];
		$stp = $parent['statut'];
	} while (($idp != 0) and ($stp != 'publie'));
	$resultats = sql_select('*','spip_biodiv_observations','id_espece='.intval($ido));
	$observations = array();
	while($res = sql_fetch($resultats)) {
		$observations[] = $idobs = $res['id_observation'];
		 spip_log("POST-EDITION Espèce $ido >> Observation $idobs","biodiv."._LOG_INFO_IMPORTANTE);		 
	}
	if($n=count($observations)) {
		sql_update(
			'spip_biodiv_observations',
			array(
				'id_espece' => 0,
				'id_famille' => intval($idr)
				),
			sql_in('id_observation',$observations)
			);
		spip_log("POST-EDITION Espèce $ido >> $n Observations déplacées dans rubrique $idr","biodiv."._LOG_INFO_IMPORTANTE);
	}
}

/**
 * Rechercher des logos par défaut dans data/logo/<objet>.jpg
 */
function biodiv_quete_logo_objet($flux) {
	if(empty($flux['data'])) {
		$objet = $flux['args']['objet'];
		$ido = $flux['args']['id_objet'];
		spip_log("LOGO-DEFAULT: $objet $ido","biodiv."._LOG_INFO_IMPORTANTE);
		if( ($chemin_image = find_in_path("data/logo/$objet.jpg")) ||
			($chemin_image = find_in_path("data-dist/logo/$objet.jpg"))
			) {
			$flux['data'] = array(
				'chemin' => $chemin_image,
				'timestamp' => @filemtime($chemin_image),
				);
			foreach (['id_document','fichier', 'titre', 'descriptif', 'credits', 'alt'] as $champ) {
				$flux['data'][$champ] = '';
			}
		}
	}
	return $flux;
}

?>