<?php

/*******************************************************************************\
 * BIODIV, plugin et squelette pour SPIP - https://www.spip.net/               *
 *         dédié à la gestion d'observations naturalistes                      *
 *                                                                             *
 *         Copyright (C) 2008-2020 Renaud LAURETTE                             *
 *                                                                             *
 * BIODIV a été développé initialement pour le projet Biodiv.Balma de l'APCVEB *
 * (Association de Protection du Cadre de Vie et de l'Environnement balmanais) * 
 *     voir Biodiv.Balma : https://balma.biodiv.fr/                            *
 *     voir APCVEB : http://apcveb.free.fr/                                    *
 *                                                                             *
 *  Ce programme est un logiciel libre distribué sous licence GNU/GPL.         *
 *  Pour plus de détails voir les fichier COPYING.txt et LICENCE-BIODIV.md     *
\*******************************************************************************/

// biodiv:inc/jws_charger_donnees_v2.php
// Interface ObsFive

include_spip('inc/jws_connect');
include_spip('action/editer_objet');
include_spip('inc/obsutil');
include_spip('action/ajouter_documents');

/*
 * Vérification du contenu minimal d'une requête
 * et de la validité de la clé associée
 */
function jws_controler_donnees_observation(&$data, &$diag) {
	// il faut une clef de session	
	if(!isset($data['session'])) {
		$diag['error'] = "Pas de session";
		return false;
	}
	// dans notre cas la session est le token d'autentification
	// Le token est-il valide ?
	$reponse = jws_controle_token($data['session']);
	$diag = $reponse;
	if($reponse['status'] <= 0) return false;
	// on loge le user identifié
	if( !jws_activer_session($reponse['status'])) {
		$diag['error'] = "Echec activation session";
		return false;
	}	
	// Les données requises sont-elles présentes ?
	if(    !isset($data['obsref'])
		OR !isset($data['titre'])
		) {
		$diag['error']= "Données insuffisantes";
		return false;
	}
		
	// Soit la date est présente au niveau de l'observation (v1:RecObs)
	// soit on en prend une dans les infos des images (v2:BioObs)
	if(!isset($data['date'])) {
		$data['date'] = null;
		if(isset($data['images']) and is_array($data['images'])) {			
			foreach($data['images'] as $image) {
				if(isset($image['date']) and (!$data['date'])) {
					$data['date'] = $image['date'];
				}
			}
		}
	}
	if(!$data['date']) {
		// Date courante par défaut
		$data['date'] = date("Y-m-d H:i:s");
	}		
			
	// Les conditions sont réunies
	// On insere l'identifiant du user de la clé dans les données
	$data['_user'] = $reponse['status'];
	$diag['error'] = "controle ok";
	return true;
}

/*
 * Création d'une observation au nom d'un user donné
 * Bidouille de sauvegarde de la session, écrasée par la sécurité de SPIP
 */
function jws_creer_observation($data, &$diag) {
    $masession = $GLOBALS['visiteur_session'];
    $id_observation = objet_inserer('observation');
    $GLOBALS['visiteur_session']=$masession;
    
    if($id_observation>0) {
    	// métadonnées
    	jws_extraire_donnees($data);
    	$c = array(); 
    	// $c['id_auteur'] = $data['_user'];
    	$c['titre'] = $data['titre'];
    	$c['espece'] = isset($data['espece'])?$data['espece']:$c['titre'];
    	$quantite = isset($data['quantite'])?intval((string)$data['quantite']):1;
    	$c['discret'] = isset($data['discret'])?$data['discret']:"non";
    	$c['quantite'] = ($quantite ==0 ) ? 1 : $quantite;
    	$optionnel = array(
    		'type_obs' => 'type_obs',
    		'denombrement' => 'denombrement',
    		'latitude' => 'lat',
    		'longitude' => 'lng',
    		'adresse' => 'adresse',
    		'insee' => 'insee',
    		'commune' => 'commune',
    		'description' => 'descriptif'
    		);
    	foreach($optionnel as $key => $value) {
    		if(isset($data[$key])) $c[$value] = $data[$key];
    	}
    	// date_obs et période du jour
    	$dobs = $data['date'];
    	preg_match(',^([1-2][0-9][0-9][0-9]-[0-1][0-9]-[0-3][0-9]) ([0-2][0-9]),',
    		$dobs,$matches);
    	$c['date_obs'] = $matches[1];
    	$c['pjour'] = periode_du_jour($matches[2]);
    	
    	$diag = $c;
    	$err = objet_modifier('observation',$id_observation,$c);
    }
    return $id_observation;
}

/*
 * Interprétation de certains champs qui ne correspondent pas
 * directement au modèle de données Biodiv
 */
function jws_extraire_donnees(&$data) {
	$data['type_obs']='autre';
	// La quantité peut être indiquée dans le champ 'data'
	if(isset($data['quantite'])) {
		// on suppose que le champ data contient uniquement la quantité
		// a remplacer par un preg pour plus de sécurité
		$q = intval((string)$data['quantite']);
		if($q > 0)  {
			$data['quantite']=$q;
			$data['type_obs']='espece';
		}
	}
	
	// La position peut être dans l'observation ou dans une des images
	if(!isset($data['latitude']) or !isset($data['longitude'])) {
		$pos = false;
		if(isset($data['images']) and is_array($data['images'])) {			
			foreach($data['images'] as $image) {
				if(!$pos
					and isset($image['latitude']) 
					and isset($image['longitude'])
					) {
					$pos = true;
					$data['latitude'] = $image['latitude'];
					$data['longitude'] = $image['longitude'];
					if(isset($image['adresse'])) 
						$data['adresse'] = $image['adresse'];
					if(isset($image['commune'])) 
						$data['commune'] = $image['commune'];
					if(isset($image['adresse'])) 
						$data['insee'] = $image['insee'];
				}
			}
		}
	}
}

/*
 * Attachement d'un fichier de données
 * Initialement une image, mais potentiellement un son également
 * L'extension détermine le type de données
 */
function jws_attacher_fichier($id_observation,$image,$rank,$ext='jpg') {
	$mimes = array(
		"jpg" => "image/jpeg",
		"3gp" => "video/3gp"	/* un son en réalité */
		);
	$docid = 0;
	$obs = sql_fetsel(array('titre'),'spip_biodiv_observations','id_observation='.sql_quote($id_observation));
	if(($obs) AND strlen($image)) {
		$imgData = base64_decode(str_replace(array('-', '_',' ','\n'), array('+', '/','+',' '), $image));
    	$upload_dir = _DIR_TMP . "upload";
    	if(!file_exists($upload_dir)) mkdir($upload_dir);
    	$upload_dir .= "/";
    	$file = $upload_dir . "o$id_observation-$rank.$ext";
    	$success = file_put_contents($file, $imgData);
    	if($success) {
    		$fcible = radical_cible_observation($id_observation,$rank) . ".$ext";
    		$download = array(
    			'name' => $fcible,
    			'tmp_name' => $file,
    			'type' => $mimes[$ext],
    			'size' => filesize($file),
    			'error' => UPLOAD_ERR_OK
    			);
    			
    		$ajouter_un_document = charger_fonction('ajouter_un_document', 'action');
    		$docid = $ajouter_un_document('new',$download,'observation',$id_observation,'document');
    		spip_log("JWS::ChargerDonneesV2 -- Reception fichier $ext $docid","jws."._LOG_INFO_IMPORTANTE);
    		// ---- spip_unlink($download['tmp_name']);
    		// Après création du doc, on donne comme titre celui de l'observation
    		// et comme crédits le nom de l'utilisateur
    		$nomAuteur = $GLOBALS['visiteur_session']['nom'];
    		sql_updateq('spip_documents',
    			array('credits' => $nomAuteur, 'titre' => $obs['titre']),
    			"id_document=".intval($docid)
    			);	
    		spip_log("JWS::ChargerDonneesV2 -- Maj document $docid : credits='".$nomAuteur."' titre='".$obsdata['titre']."'","jws."._LOG_INFO_IMPORTANTE);
    	} else {
    		spip_log("JWS::ChargerDonneesV2 -- Echec creation fichier $ext $rank : observation $id_observation","jws."._LOG_INFO_IMPORTANTE);
    	}
    } else {
    	
    	spip_log("JWS::ChargerDonneesV2 -- Probleme avec fichier $ext $rank : observation $id_observation","jws."._LOG_INFO_IMPORTANTE);
    }
    return $docid;		
}
?>