<?php
/**
 * Fichier gérant l'installation et désinstallation du plugin Biodiv
 *
 * @plugin     Biodiv
 * @copyright  2020
 * @author     Renaud LAURETTE
 * @licence    GNU/GPL
 * @package    SPIP\Biodiv\Installation
 */

if (!defined('_ECRIRE_INC_VERSION')) {
	return;
}

include_spip('action/editer_objet');

/**
 * Fonction d'installation et de mise à jour du plugin Biodiv.
 *
 * Vous pouvez :
 *
 * - créer la structure SQL,
 * - insérer du pre-contenu,
 * - installer des valeurs de configuration,
 * - mettre à jour la structure SQL 
 *
 * @param string $nom_meta_base_version
 *     Nom de la meta informant de la version du schéma de données du plugin installé dans SPIP
 * @param string $version_cible
 *     Version du schéma de données dans ce plugin (déclaré dans paquet.xml)
 * @return void
**/
function biodiv_upgrade($nom_meta_base_version, $version_cible) {
	$maj = array();
	# quelques exemples
	# (que vous pouvez supprimer !)
	# 
	# $maj['create'] = array(array('creer_base'));
	#
	# include_spip('inc/config')
	# $maj['create'] = array(
	#	array('maj_tables', array('spip_xx', 'spip_xx_liens')),
	#	array('ecrire_config', 'biodiv', array('exemple' => "Texte de l'exemple"))
	#);
	#
	# $maj['1.1.0']  = array(array('sql_alter','TABLE spip_xx RENAME TO spip_yy'));
	# $maj['1.2.0']  = array(array('sql_alter','TABLE spip_xx DROP COLUMN id_auteur'));
	# $maj['1.3.0']  = array(
	#	array('sql_alter','TABLE spip_xx CHANGE numero numero int(11) default 0 NOT NULL'),
	#	array('sql_alter','TABLE spip_xx CHANGE texte petit_texte mediumtext NOT NULL default \'\''),
	#   array('une_fonction_sans_arguments_definie_dans_ce_fichier')
	# );
	# ...

	// Creation : installation complète
	$maj['create'] = array(
		array('maj_tables', array('spip_biodiv_observations', 
								'spip_biodiv_observations_liens', 
								'spip_biodiv_sources', 
								'spip_taxrefs',
								'spip_biodiv_campagnes'
								))
		);

    $maj['3.0'] = array(
    	array('maj_tables', array('spip_biodiv_observations_liens', 'spip_biodiv_campagnes')),
    	array('biodiv_migrations', array('3.0')),
    	array('sql_alter','TABLE spip_biodiv_observations CHANGE id_auteur xd_auteur BIGINT(21)')
    	);
    
    $maj['3.1'] = array(
    	array('maj_tables', array('spip_taxrefs'))
    	);

    $maj['3.2'] = array(
    	array('biodiv_migrations', array('3.2'))
    	);
   
	include_spip('base/upgrade');
	maj_plugin($nom_meta_base_version, $version_cible, $maj);
	biodiv_controle_installation();
}


/**
 * Fonction de désinstallation du plugin Biodiv.
 * 
 * Vous devez :
 * - nettoyer toutes les données ajoutées par le plugin et son utilisation
 * - supprimer les tables et les champs créés par le plugin. 
 *
 * @param string $nom_meta_base_version
 *     Nom de la meta informant de la version du schéma de données du plugin installé dans SPIP
 * @return void
**/
function biodiv_vider_tables($nom_meta_base_version) {

	sql_drop_table('spip_biodiv_observations');
	sql_drop_table('spip_biodiv_observations_liens');
	sql_drop_table('spip_biodiv_sources');
	@sql_drop_table('spip_taxons');
	sql_drop_table('spip_taxrefs');
	sql_drop_table('spip_biodiv_campagnes');

	# Nettoyer les liens courants (le génie optimiser_base_disparus se chargera de nettoyer toutes les tables de liens)
	sql_delete('spip_documents_liens', sql_in('objet', array('observation', 'source', 'binomial', 'campagne')));
	sql_delete('spip_mots_liens', sql_in('objet', array('observation', 'source', 'binomial', 'campagne')));
	sql_delete('spip_auteurs_liens', sql_in('objet', array('observation', 'source', 'binomial', 'campagne')));
	# Nettoyer les versionnages et forums
	sql_delete('spip_versions', sql_in('objet', array('observation', 'source', 'binomial', 'campagne')));
	sql_delete('spip_versions_fragments', sql_in('objet', array('observation', 'source', 'binomial', 'campagne')));
	sql_delete('spip_forum', sql_in('objet', array('observation', 'source', 'binomial', 'campagne')));

	effacer_meta($nom_meta_base_version);
}



/**
 * Fonction de vérification des installations spécifiques
 * 
 * @param void
 * @return void
**/
function biodiv_controle_installation() {

	// Chargement des définitions
	include_spip('base/biodiv_init');
		
	// Vérifier l'existance de la source principale (N°1)
	if(!sql_countsel('spip_biodiv_sources')) {
		$champs = array(
			'acces' => lire_config('/meta/adresse_site'),
			'titre' => lire_config('/meta/nom_site'),
			'descriptif' => 'Base des observations déposées directement sur ce site',
			'support' => 'Base de données', 
			'statut_source' => 'Te',
			'createur' => '(installation automatique)',
			'publication' => date('Y-m-d')
			);		
		$id = objet_inserer('source');
		if (!($id = intval($id)) > 0) {
			spip_log("Installation - ECHEC Insertion de la source","biodiv."._LOG_INFO_IMPORTANTE);
		} else {
			$err = objet_modifier('source',$id,$champs);
			spip_log("Installation - Modification de la source $id : $err","biodiv."._LOG_INFO_IMPORTANTE);
		}
	}	
	
	// Déclaration des types de fichiers spécifiques à Biodiv
	$bioTypes = biodiv_types_specifiques();
	foreach($bioTypes as $btype) {
		if(!sql_countsel('spip_types_documents', "extension=".sql_quote($btype['extension']))) {
			sql_insertq('spip_types_documents', $btype);
			spip_log("Installation - declaration type document ".$btype['titre'],
					"biodiv."._LOG_INFO_IMPORTANTE);
		}	
	}
	
	// Créer les répertoires spécifiques
	$repertoires = biodiv_repertoires_tmp();
	foreach($repertoires as $rep) {
		$repertoire = _DIR_TMP . $rep;
		if(!is_dir($repertoire)) {
			if(!file_exists($repertoire)) {
				mkdir($repertoire);
				spip_log("Installation - Creation repertoire $repertoire","biodiv."._LOG_INFO_IMPORTANTE);
			} else {
				spip_log("Installation : impossible de creer repertoire '$repertoire' : un fichier du meme nom existe deja.", "biodiv."._LOG_INFO_IMPORTANTE);
			}
		}
	}
	
	// S'assurer des configurations minimales
	ecrire_meta('article_soustitre','oui');			// Pour le nom latin des espèces
	ecrire_meta('article_descriptif','oui');		// Pour les autres noms des espèces
	ecrire_meta('accepter_visiteurs','oui');		// Pour l'inscription des observateurs
	
	// S'assurer de la possibilité de joindre des documents à certains objets
	$objets = lire_config('/meta/documents_objets');
	$ajouts = "";
	foreach(array('spip_biodiv_observations','spip_biodiv_campagnes') as $table) {
		if(!strpos($objets,$table)) $ajouts .= "$table,";
	}
	if(strlen($ajouts)) {
		$objets .= $ajouts;
		ecrire_meta('documents_objets',$objets);
	}
}



/**
 * Fonction générique de traitement des migrations complexes
 * 
 * @param string $version	Version cible du schéma 
 * @return void
**/
function biodiv_migrations($args) {
	spip_log("Migrations: args ".print_r($args,true), "biodiv."._LOG_INFO_IMPORTANTE);
	$version = $args[0];
	switch($version) {
		
	case '3.0':
		include_spip('base/biodiv_migration_3_0');
		biodiv_migration_3_0();
		break;	
		
	case '3.2':
		// Les paramètres de cartographie se déplacent de Biodiv à Feuillet
		// On récupère les anciennes valeurs pour initialiser Feuillet
		ecrire_config('feuillet/lat',lire_config('biodiv/lat'));
		ecrire_config('feuillet/lng',lire_config('biodiv/lng'));
		ecrire_config('feuillet/zoom',lire_config('biodiv/zoom'));
		ecrire_config('feuillet/tuiles',lire_config('biodiv/tuiles'));
		break;

	default:
		break;
	}
}
		
