<?php

/*******************************************************************************\
 * BIODIV, plugin et squelette pour SPIP - https://www.spip.net/               *
 *         dédié à la gestion d'observations naturalistes                      *
 *                                                                             *
 *         Copyright (C) 2008-2020 Renaud LAURETTE                             *
 *                                                                             *
 * BIODIV a été développé initialement pour le projet Biodiv.Balma de l'APCVEB *
 * (Association de Protection du Cadre de Vie et de l'Environnement balmanais) * 
 *     voir Biodiv.Balma : https://balma.biodiv.fr/                            *
 *     voir APCVEB : http://apcveb.free.fr/                                    *
 *                                                                             *
 *  Ce programme est un logiciel libre distribué sous licence GNU/GPL.         *
 *  Pour plus de détails voir les fichier COPYING.txt et LICENCE-BIODIV.md     *
\*******************************************************************************/

// biodiv:inc/obsutil.php

define('_TAILLE_MAX_OBSERVATION', 1048576); // 1 Mo

// gestion des extensions autorisées pour les fichiers joints aux observations
function documenter_observation_formats() {
	// les clefs sont les extensions acceptées en entrée qui seront converties en valeurs
	$formats = array(
		'jpg'  => 'jpg',
		'jpeg' => 'jpg',
		'jpe'  => 'jpg',
		'png'  => 'png',
		'gif'  => 'gif',
		'mp3'  => 'mp3'
		);
	return $formats;
}

// Taille maximale de fichier autorisée en téléchargement en fonction de l'extension

function taille_max_format($ext) 
{
	switch($ext) {
	case 'jpg':
	case 'png':
	case 'gif':
		return _TAILLE_MAX_OBSERVATION;
	case 'mp3':
		return 2*_TAILLE_MAX_OBSERVATION;
	default:
		return 0;
	}
}

// Génère un radical de nom de fichier pour une observation
// basé sur l'identifiant de l'observation et un compteur
function radical_cible_observation($idobs,$nbfich) {
	$cible = "obs_" . $idobs . "_" . $nbfich;
	return $cible;
}

// Sépare le radical et l'extension d'un nom de fichier
// avec ou sans chemin en entrée
function radical_et_extension($file)
{
  $fichier = basename($file);
  $ext = substr($fichier,strrpos($fichier,".")+1);
  $radical = substr($fichier,0,strrpos($fichier,"."));
  return array($radical, $ext);
}


// retourne une chaine donnant la période du jour en fonction de l'heure
function periode_du_jour($heure) 
{
  if(intval($heure) < 5) {
    $value = 'nuit';
  } elseif (intval($heure) < 7) {
    $value = 'lever du soleil';
  } elseif(intval($heure) < 12) {
    $value = 'matin';
  } elseif(intval($heure) < 19) {
    $value = 'apres-midi';
  } elseif(intval($heure) < 22) {
    $value = 'coucher du soleil';
  } else {
    $value = 'nuit';
  }
  return $value;
}

// Traite une chaine contenant une date EXIF et en extrait les différents composants
// pour les transformer en une date au format biodiv (mysql) et en une période du jour
// retourne un array ['date'] et ['periode'] ou null
function date_et_periode_depuis_exif($date_exif) 
{
   $detp = array();
   $pattern=",(\d{4})\:(\d{2})\:(\d{2}) (\d{2})\:\d{2}\:\d{2},";
   if(preg_match($pattern,$date_exif,$parts)) {
     $detp['date'] = $parts[1]."-".$parts[2]."-".$parts[3];
     $detp['periode'] = periode_du_jour($parts[4]);
     return $detp;
   }
   return null;
}

// Controle de la validité d'un fichier à joindre à une observation
// Le fichier doit être lisible
// Sa taille est limitée
// Son extension doit être valide
// Retourne l'extension à utiliser si ok, null sinon
function controle_fichier_observation($chemin_complet)
{
  // lisibilité
  if(!is_readable($chemin_complet)) return array("Fichier illisible. ",null);
  // extension
  list($radical,$extension) = radical_et_extension($chemin_complet);
  $extok = documenter_observation_formats();
  if(in_array($extension, array_keys($extok))) return array(null, $extok[$extension]);
  // échec
  return array("Extension '$extension' interdite. ",null);
  // taille
  if(filesize($chemin_complet) > taille_max_format($extok[$extension])) return array("Fichier trop gros. ", null);

}



?>