<?php
/**
 * Définit les autorisations du plugin Biodiv
 *
 * @plugin     Biodiv
 * @copyright  2020-2025
 * @author     Renaud LAURETTE
 * @licence    GNU/GPL
 * @package    SPIP\Biodiv\Autorisations
 */

if (!defined('_ECRIRE_INC_VERSION')) {
	return;
}


/*
 * Un fichier d'autorisations permet de regrouper
 * les fonctions d'autorisations de votre plugin
 */

/**
 * Fonction d'appel pour le pipeline
 * @pipeline autoriser */
function biodiv_autoriser() {
}


/* Exemple
function autoriser_biodiv_configurer_dist($faire, $type, $id, $qui, $opt) {
	// type est un objet (la plupart du temps) ou une chose.
	// autoriser('configurer', '_biodiv') => $type = 'biodiv'
	// au choix :
	return autoriser('webmestre', $type, $id, $qui, $opt); // seulement les webmestres
	return autoriser('configurer', '', $id, $qui, $opt); // seulement les administrateurs complets
	return $qui['statut'] == '0minirezo'; // seulement les administrateurs (même les restreints)
	// ...
}
*/

// -----------------
// Objet biodiv_observations


/**
 * Autorisation de voir un élément de menu (biodivobservations)
 *
 * @param  string $faire Action demandée
 * @param  string $type  Type d'objet sur lequel appliquer l'action
 * @param  int    $id    Identifiant de l'objet
 * @param  array  $qui   Description de l'auteur demandant l'autorisation
 * @param  array  $opt   Options de cette autorisation
 * @return bool          true s'il a le droit, false sinon
**/
function autoriser_biodivobservations_menu_dist($faire, $type, $id, $qui, $opt) {
	return true;
}


/**
 * Autorisation de créer (observation)
 *
 * @param  string $faire Action demandée
 * @param  string $type  Type d'objet sur lequel appliquer l'action
 * @param  int    $id    Identifiant de l'objet
 * @param  array  $qui   Description de l'auteur demandant l'autorisation
 * @param  array  $opt   Options de cette autorisation
 * @return bool          true s'il a le droit, false sinon
**/
function autoriser_observation_creer_dist($faire, $type, $id, $qui, $opt) {
	return in_array($qui['statut'], array('0minirezo', '1comite', '6forum'));
}

/**
 * Autorisation de voir (observation)
 *
 * @param  string $faire Action demandée
 * @param  string $type  Type d'objet sur lequel appliquer l'action
 * @param  int    $id    Identifiant de l'objet
 * @param  array  $qui   Description de l'auteur demandant l'autorisation
 * @param  array  $opt   Options de cette autorisation
 * @return bool          true s'il a le droit, false sinon
**/
function autoriser_observation_voir_dist($faire, $type, $id, $qui, $opt) {
	return true;
}

/**
 * Autorisation de modifier (observation)
 *
 * @param  string $faire Action demandée
 * @param  string $type  Type d'objet sur lequel appliquer l'action
 * @param  int    $id    Identifiant de l'objet
 * @param  array  $qui   Description de l'auteur demandant l'autorisation
 * @param  array  $opt   Options de cette autorisation
 * @return bool          true s'il a le droit, false sinon
**/
function autoriser_observation_modifier_dist($faire, $type, $id, $qui, $opt) {
	if($id == 0) {
		// nouvel objet : autorisé à tous les utilisateurs loggés.
		return in_array ($qui['statut'], array('0minirezo', '1comite', '6forum'));
	} else {
		// objet existant. Ok si propriétaire ou admin
		$autorisation = in_array ($qui['statut'], array('0minirezo'));
		if(!$autorisation) {
			$liens = sql_allfetsel("id_auteur", 
						"spip_auteurs_liens", 
						array(
							"objet = ".sql_quote("observation"),
							"id_objet = ".intval($id)
							));
			foreach($liens as $lien) {	
				$a = (intval($qui['id_auteur']) == intval($lien['id_auteur']))?true:false;
				$autorisation = $autorisation || $a;
			}
		}		
		return $autorisation;
	}
}

/**
 * Autorisation de supprimer (observation)
 *
 * @param  string $faire Action demandée
 * @param  string $type  Type d'objet sur lequel appliquer l'action
 * @param  int    $id    Identifiant de l'objet
 * @param  array  $qui   Description de l'auteur demandant l'autorisation
 * @param  array  $opt   Options de cette autorisation
 * @return bool          true s'il a le droit, false sinon
**/
function autoriser_observation_supprimer_dist($faire, $type, $id, $qui, $opt) {
	return autoriser_observation_modifier_dist($faire, $type, $id, $qui, $opt);
}


/**
 * Autorisation d'exporter (observation)
 * Autoriser à exporter une ou des observations dans des formats cartographiques
 *
 * @param  string $faire Action demandée
 * @param  string $type  Type d'objet sur lequel appliquer l'action
 * @param  int    $id    Identifiant de l'objet
 * @param  array  $qui   Description de l'auteur demandant l'autorisation
 * @param  array  $opt   Options de cette autorisation
 * @return bool          true s'il a le droit, false sinon
**/
function autoriser_observation_exporter_dist($faire, $type, $id, $qui, $opt) {
    return accorder_selon_zone($qui,'export');
    /* return autoriser('ecrire'); */
}

/**
 * Autorisation de débusquer (observation)
 * Autoriser à afficher la localisation confidentielle d'une observation
 *
 * @param  string $faire Action demandée
 * @param  string $type  Type d'objet sur lequel appliquer l'action
 * @param  int    $id    Identifiant de l'objet
 * @param  array  $qui   Description de l'auteur demandant l'autorisation
 * @param  array  $opt   Options de cette autorisation
 * @return bool          true s'il a le droit, false sinon
**/
function autoriser_observation_debusquer_dist($faire, $type, $id, $qui, $opt) {
	
	if($id == 0) {
		// autorisation générale 
		// return in_array ($qui['statut'], array('0minirezo'));  // administrateurs seulement.
	    return
		 (  in_array($qui['statut'], array('0minirezo')) // les administrateurs voient tout 
		 || accorder_selon_zone($qui,'confidentiel'));	 // les membres de la zone 'biodiv.confidentiel' aussi
	} else {
		// autorisation sur une Obs spécifique.
		// l'administrateur a tous les droits ; les membres de la zone 'biodiv.confidentiel' aussi
		$autorisation = (($qui['statut'] == '0minirezo') || accorder_selon_zone($qui,'confidentiel'));
		// les auteurs de l'observation aussi
		$liens = sql_allfetsel("id_auteur", 
						"spip_auteurs_liens", 
						array(
							"objet=".sql_quote("observation"),
							"id_objet=".sql_quote($id)
							));
		foreach($liens as $lien) {
				$autorisation = $autorisation || ($qui['id_auteur'] == $lien['id_auteur']);
		}
		// Les privilégiés ont le droit
		if($autorisation) { return true; }	
		
		// voyons maintenant les autres cas
		$r = sql_fetsel("discret,id_espece", "spip_biodiv_observations", "id_observation=".sql_quote($id));
		$t = sql_fetsel("sensible,cd_ref","spip_taxrefs", "id_espece=".sql_quote($r['id_espece']));
		if($r['discret'] == "oui") {
			// l'observation est confidentielle : pas d'accès
			return false;
		}
		if($t) { 
			// L'observation est liée à un taxon
			// Si le taxon est sensible, pas d'autorisation
			return ($t['sensible'] !='oui');
		}
		// On a tout vérifié. Pas de restriction
		return true;
	}
}

/**
 * Filtre une liste d'observations pour ne retenir que celles
 * qui sont débusquables par l'utilisateur considéré.
 * Cette fonction recalcule directement les droits, pour éviter
 * un appel récurrent à autoriser_observation_debusquer
 * Cette fonction est appelée par le filtre |debusquable
 *
 * @param  array $tabobs Tableau d'identifiants d'observations demandées
 * @param  array $qui    Description de l'auteur demandant l'autorisation
 * @return array         Tableau d'identifiants d'observations autorisées
**/
function filtrer_observations_debusquables($tabobs,$qui) {
	// spip_log("Debusquer: ".print_r($qui,true), _LOG_INFO_IMPORTANTE);
	
	if( is_array($qui) && isset($qui['statut']) &&
		(   in_array($qui['statut'], array('0minirezo')) // les administrateurs voient tout 
		 || accorder_selon_zone($qui,'confidentiel')	 // les membres de la zone 'biodiv.confidentiel' aussi
		 )
		) {
		return $tabobs;         
	}
	
	/* on récupère les infos sur les observations ... */
	$lobs = sql_in('id_observation',$tabobs);
	$r = sql_allfetsel("id_observation,discret,id_espece", "spip_biodiv_observations", $lobs);
	
	/* ... pour identifier les observations discretes */
	$dr = array_filter($r, function($a) { return ($a['discret'] == "oui"); });
	$discret = array_map( function($a) { return $a['id_observation']; }, $dr);
	// spip_log("Debusquer: discret: ".print_r($discret,true), _LOG_INFO_IMPORTANTE);
	
	/* ... puis les especes sensibles */
	$espece = array_map( function($a) { return $a['id_espece']; }, $r);
	$espece = array_values(array_unique($espece));
	$lesp = sql_in('id_espece',$espece);
	$t = sql_allfetsel("id_espece","spip_taxrefs", array("sensible='oui'", $lesp));	
	$es = array_map( function($a) { return $a['id_espece']; }, $t);
	// spip_log("Debusquer: especes sensibles: ".print_r($es,true), _LOG_INFO_IMPORTANTE);
	
	/* ... et en déduire les observations sensibles */
	$ds = array_filter($r, function($a) { return in_array($a['id_espece'],$es); });
	$sensible = array_map( function($a) { return $a['id_observation']; }, $ds);
	// spip_log("Debusquer: observations sensibles: ".print_r($sensible,true), _LOG_INFO_IMPORTANTE);
	
	/* les observations interdites sont donc les discretes ou sensibles */
	$interdit = array_values(array_merge($sensible,$discret));
	// spip_log("Debusquer: observations interdites: ".print_r($sensible,true), _LOG_INFO_IMPORTANTE);
	
	/* sauf si on en est l'auteur ... */
	$sauf = array();
	if(is_array($qui) && isset($qui['id_auteur'])) {
		$df = sql_allfetsel("id_objet", "spip_auteurs_liens", array(
					"objet=".sql_quote("observation"), 
					"id_auteur=".sql_quote($qui['id_auteur']),
					sql_in('id_objet',$interdit)
					));
		$sauf = array_map( function($a) { return $a['id_objet']; }, $df);
	}
	// spip_log("Debusquer: observations persos: ".print_r($sauf,true), _LOG_INFO_IMPORTANTE);
	
	/* il ne reste plus qu'à soustraire ... */
	$interdit = array_values(array_diff($interdit,$sauf));
	$permis = array_values(array_diff($tabobs,$interdit));

	// spip_log("Debusquer: observations publiques: ".print_r($permis,true), _LOG_INFO_IMPORTANTE);
	return $permis;
}



/**
 * Recherche d'une autorisation selon un statut
 * ou l'appartenance à une zone biodiv donnée
 * 
 * @param  array  $qui   	Description de l'auteur demandant l'autorisation
 * @param  string $nomZone  Nom de la zone Biodiv
 * @return bool          	true s'il a le droit, false sinon
**/
function accorder_selon_zone($qui,$nomZone) {

	if( (!isset($qui)) || (!is_array($qui)) || (!isset($qui['statut'])) ) {
		return false;
	}
	
	if(in_array ($qui['statut'], array('0minirezo', '1comite'))) {
		// ceux-ci ont toujours le droit
		return true;
	}
	
	// pour les autres on discute ...
	if($qui['id_auteur']) {
		// On teste si acces_restreint est installé
		$pu = unserialize($GLOBALS['meta']['plugin_installes']);	
		if(in_array('acces_restreint',$pu)) {
			// acces_restreint est là. Y a-t-il une zone 'biodiv.recherche' ?
			$zone = sql_getfetsel("id_zone","spip_zones","titre='biodiv.$nomZone'");
	  		// echo "<p>Zone de recherche : $zone</p>";
	  		if(accesrestreint_acces_zone(intval($zone),$qui['id_auteur'])) {
	  			// La zone existe et l'utilisateur est déclaré dedans : il a le droit
	  			return true;
	  		}
	  	}
	}
	// pas bon
	return false;  
}

/**
 * Autorisation de typer (observation)
 * Autoriser à changer le type d'une observation et à créer une observation
 * d'un type différent de 'espece'.
 *
 * @param  string $faire Action demandée
 * @param  string $type  Type d'objet sur lequel appliquer l'action
 * @param  int    $id    Identifiant de l'objet
 * @param  array  $qui   Description de l'auteur demandant l'autorisation
 * @param  array  $opt   Options de cette autorisation
 * @return bool          true s'il a le droit, false sinon
**/
function autoriser_observation_typer_dist($faire, $type, $id, $qui, $opt) {
	return autoriser_observation_creer_dist($faire, $type, $id, $qui, $opt);
   /* return autoriser('ecrire'); */
}


/**
 * Autorisation à rechercher (observation)
 * Autoriser un utilisateur à accéder aux fonctions de recherche avancées
 * telles que la recherche par atributs et zones géographiques.
 * Par défaut réservé aux administrateurs, mais, via le plugin 'acces_restreint'
 * l'autorisation est étendue aux membres de la zone 'biodiv.recherche'
 *
 * @param  string $faire Action demandée
 * @param  string $type  Type d'objet sur lequel appliquer l'action
 * @param  int    $id    Identifiant de l'objet
 * @param  array  $qui   Description de l'auteur demandant l'autorisation
 * @param  array  $opt   Options de cette autorisation
 * @return bool          true s'il a le droit, false sinon
**/

function autoriser_observation_rechercher_dist($faire, $type, $id, $qui, $opt) {
	// On ne teste pas $id : l'autorisation est de portée générale	
	return accorder_selon_zone($qui,'recherche');

	/*
	if(in_array ($qui['statut'], array('0minirezo', '1comite'))) {
		// ceux-ci ont toujours le droit
		return true;
	}
	
	// pour les autres on discute ...
	if($qui['id_auteur']) {
		// On teste si acces_restreint est installé
		$pu = unserialize($GLOBALS['meta']['plugin_installes']);	
		if(in_array('acces_restreint',$pu)) {
			// acces_restreint est là. Y a-t-il une zone 'biodiv.recherche' ?
			$zone = sql_getfetsel("id_zone","spip_zones","titre='biodiv.recherche'");
	  		// echo "<p>Zone de recherche : $zone</p>";
	  		if(accesrestreint_acces_zone(intval($zone),$qui['id_auteur'])) {
	  			// La zone existe et l'utilisateur est déclaré dedans : il a le droit
	  			return true;
	  		}
	  	}
	}
	// pas bon
	return false;   
*/
}


/**
 * Autorisation de lier/délier l'élément (biodivobservations)
 *
 * @param  string $faire Action demandée
 * @param  string $type  Type d'objet sur lequel appliquer l'action
 * @param  int    $id    Identifiant de l'objet
 * @param  array  $qui   Description de l'auteur demandant l'autorisation
 * @param  array  $opt   Options de cette autorisation
 * @return bool          true s'il a le droit, false sinon
**/
function autoriser_associerbiodivobservations_dist($faire, $type, $id, $qui, $opt) {
	return $qui['statut'] == '0minirezo' /* and !$qui['restreint'] */;
}

// -----------------
// Objet biodiv_sources


/**
 * Autorisation de voir un élément de menu (biodivsources)
 *
 * @param  string $faire Action demandée
 * @param  string $type  Type d'objet sur lequel appliquer l'action
 * @param  int    $id    Identifiant de l'objet
 * @param  array  $qui   Description de l'auteur demandant l'autorisation
 * @param  array  $opt   Options de cette autorisation
 * @return bool          true s'il a le droit, false sinon
**/
function autoriser_biodivsources_menu_dist($faire, $type, $id, $qui, $opt) {
	return true;
}


/**
 * Autorisation de créer (source)
 *
 * @param  string $faire Action demandée
 * @param  string $type  Type d'objet sur lequel appliquer l'action
 * @param  int    $id    Identifiant de l'objet
 * @param  array  $qui   Description de l'auteur demandant l'autorisation
 * @param  array  $opt   Options de cette autorisation
 * @return bool          true s'il a le droit, false sinon
**/
function autoriser_source_creer_dist($faire, $type, $id, $qui, $opt) {
	return $qui['statut'] == '0minirezo';
}

/**
 * Autorisation de voir (source)
 *
 * @param  string $faire Action demandée
 * @param  string $type  Type d'objet sur lequel appliquer l'action
 * @param  int    $id    Identifiant de l'objet
 * @param  array  $qui   Description de l'auteur demandant l'autorisation
 * @param  array  $opt   Options de cette autorisation
 * @return bool          true s'il a le droit, false sinon
**/
function autoriser_source_voir_dist($faire, $type, $id, $qui, $opt) {
	return true;
}

/**
 * Autorisation de modifier (source)
 *
 * @param  string $faire Action demandée
 * @param  string $type  Type d'objet sur lequel appliquer l'action
 * @param  int    $id    Identifiant de l'objet
 * @param  array  $qui   Description de l'auteur demandant l'autorisation
 * @param  array  $opt   Options de cette autorisation
 * @return bool          true s'il a le droit, false sinon
**/
function autoriser_source_modifier_dist($faire, $type, $id, $qui, $opt) {
	return $qui['statut'] == '0minirezo';
}

/**
 * Autorisation de supprimer (source)
 *
 * @param  string $faire Action demandée
 * @param  string $type  Type d'objet sur lequel appliquer l'action
 * @param  int    $id    Identifiant de l'objet
 * @param  array  $qui   Description de l'auteur demandant l'autorisation
 * @param  array  $opt   Options de cette autorisation
 * @return bool          true s'il a le droit, false sinon
**/
function autoriser_source_supprimer_dist($faire, $type, $id, $qui, $opt) {
	return $qui['statut'] == '0minirezo' and !$qui['restreint'];
}


// -----------------
// Gestion des ordres par lots

/**
 * Autorisation d'exécuter des ordres par lots
 *
 * @param  string $faire Action demandée
 * @param  string $type  Type d'objet sur lequel appliquer l'action
 * @param  int    $id    Identifiant de l'objet
 * @param  array  $qui   Description de l'auteur demandant l'autorisation
 * @param  array  $opt   Options de cette autorisation
 * @return bool          true s'il a le droit, false sinon
**/

function autoriser_ordonner_dist($faire, $type, $id, $qui, $opt)
{
	return in_array ($qui['statut'], array('0minirezo'));	
}


// -----------------
// Gestion des groupes de mots clés

/**
 * Autoriser a modifier un groupe de mots $id
 * y compris en ajoutant/modifiant les mots lui appartenant
 * Surcharge du standard pour étendre aux administrateurs restreints
 *
 * @param  string $faire Action demandée
 * @param  string $type  Type d'objet sur lequel appliquer l'action
 * @param  int    $id    Identifiant de l'objet
 * @param  array  $qui   Description de l'auteur demandant l'autorisation
 * @param  array  $opt   Options de cette autorisation
 * @return bool          true s'il a le droit, false sinon
**/
function autoriser_groupemots_modifier($faire, $type, $id, $qui, $opt) {
	return
		$qui['statut'] == '0minirezo';
}

// -----------------
// Objet taxons


/**
 * Autorisation de voir un élément de menu (taxons)
 *
 * @param  string $faire Action demandée
 * @param  string $type  Type d'objet sur lequel appliquer l'action
 * @param  int    $id    Identifiant de l'objet
 * @param  array  $qui   Description de l'auteur demandant l'autorisation
 * @param  array  $opt   Options de cette autorisation
 * @return bool          true s'il a le droit, false sinon
**/
function autoriser_taxons_menu_dist($faire, $type, $id, $qui, $opt) {
	return true;
}


/**
 * Autorisation de créer (binomial)
 *
 * @param  string $faire Action demandée
 * @param  string $type  Type d'objet sur lequel appliquer l'action
 * @param  int    $id    Identifiant de l'objet
 * @param  array  $qui   Description de l'auteur demandant l'autorisation
 * @param  array  $opt   Options de cette autorisation
 * @return bool          true s'il a le droit, false sinon
**/
function autoriser_binomial_creer_dist($faire, $type, $id, $qui, $opt) {
	return $qui['statut'] == '0minirezo' and !$qui['restreint'];
}

/**
 * Autorisation de voir (binomial)
 *
 * @param  string $faire Action demandée
 * @param  string $type  Type d'objet sur lequel appliquer l'action
 * @param  int    $id    Identifiant de l'objet
 * @param  array  $qui   Description de l'auteur demandant l'autorisation
 * @param  array  $opt   Options de cette autorisation
 * @return bool          true s'il a le droit, false sinon
**/
function autoriser_binomial_voir_dist($faire, $type, $id, $qui, $opt) {
	return true;
}

/**
 * Autorisation de modifier (binomial)
 *
 * @param  string $faire Action demandée
 * @param  string $type  Type d'objet sur lequel appliquer l'action
 * @param  int    $id    Identifiant de l'objet
 * @param  array  $qui   Description de l'auteur demandant l'autorisation
 * @param  array  $opt   Options de cette autorisation
 * @return bool          true s'il a le droit, false sinon
**/
function autoriser_binomial_modifier_dist($faire, $type, $id, $qui, $opt) {
	return $qui['statut'] == '0minirezo' and !$qui['restreint'];
}

/**
 * Autorisation de supprimer (binomial)
 *
 * @param  string $faire Action demandée
 * @param  string $type  Type d'objet sur lequel appliquer l'action
 * @param  int    $id    Identifiant de l'objet
 * @param  array  $qui   Description de l'auteur demandant l'autorisation
 * @param  array  $opt   Options de cette autorisation
 * @return bool          true s'il a le droit, false sinon
**/
function autoriser_binomial_supprimer_dist($faire, $type, $id, $qui, $opt) {
	return $qui['statut'] == '0minirezo' and !$qui['restreint'];
}


// -----------------
// Objet biodiv_campagnes


/**
 * Autorisation de voir un élément de menu (biodivcampagnes)
 *
 * @param  string $faire Action demandée
 * @param  string $type  Type d'objet sur lequel appliquer l'action
 * @param  int    $id    Identifiant de l'objet
 * @param  array  $qui   Description de l'auteur demandant l'autorisation
 * @param  array  $opt   Options de cette autorisation
 * @return bool          true s'il a le droit, false sinon
**/
function autoriser_biodivcampagnes_menu_dist($faire, $type, $id, $qui, $opt) {
	return true;
}


/**
 * Autorisation de créer (campagne)
 *
 * @param  string $faire Action demandée
 * @param  string $type  Type d'objet sur lequel appliquer l'action
 * @param  int    $id    Identifiant de l'objet
 * @param  array  $qui   Description de l'auteur demandant l'autorisation
 * @param  array  $opt   Options de cette autorisation
 * @return bool          true s'il a le droit, false sinon
**/
function autoriser_campagne_creer_dist($faire, $type, $id, $qui, $opt) {
	return (in_array($qui['statut'], array('0minirezo', '1comite')) and sql_countsel('spip_rubriques')>0);
}

/**
 * Autorisation de voir (campagne)
 *
 * @param  string $faire Action demandée
 * @param  string $type  Type d'objet sur lequel appliquer l'action
 * @param  int    $id    Identifiant de l'objet
 * @param  array  $qui   Description de l'auteur demandant l'autorisation
 * @param  array  $opt   Options de cette autorisation
 * @return bool          true s'il a le droit, false sinon
**/
function autoriser_campagne_voir_dist($faire, $type, $id, $qui, $opt) {
	return true;
}

/**
 * Autorisation de modifier (campagne)
 *
 * @param  string $faire Action demandée
 * @param  string $type  Type d'objet sur lequel appliquer l'action
 * @param  int    $id    Identifiant de l'objet
 * @param  array  $qui   Description de l'auteur demandant l'autorisation
 * @param  array  $opt   Options de cette autorisation
 * @return bool          true s'il a le droit, false sinon
**/
function autoriser_campagne_modifier_dist($faire, $type, $id, $qui, $opt) {
	return in_array($qui['statut'], array('0minirezo', '1comite'));
}

/**
 * Autorisation de supprimer (campagne)
 *
 * @param  string $faire Action demandée
 * @param  string $type  Type d'objet sur lequel appliquer l'action
 * @param  int    $id    Identifiant de l'objet
 * @param  array  $qui   Description de l'auteur demandant l'autorisation
 * @param  array  $opt   Options de cette autorisation
 * @return bool          true s'il a le droit, false sinon
**/
function autoriser_campagne_supprimer_dist($faire, $type, $id, $qui, $opt) {
	return $qui['statut'] == '0minirezo' and !$qui['restreint'];
}

/**
 * Autorisation de créer l'élément (campagne) dans une rubrique
 *
 * @param  string $faire Action demandée
 * @param  string $type  Type d'objet sur lequel appliquer l'action
 * @param  int    $id    Identifiant de l'objet
 * @param  array  $qui   Description de l'auteur demandant l'autorisation
 * @param  array  $opt   Options de cette autorisation
 * @return bool          true s'il a le droit, false sinon
**/
function autoriser_rubrique_creercampagnedans_dist($faire, $type, $id, $qui, $opt) {
	return ($id and autoriser('voir', 'rubrique', $id) and autoriser('creer', 'campagne', $id));
}


/**
 * Autorisation d'associer une observation à une campagne
 *
 * @param  string $faire Action demandée
 * @param  string $type  Type d'objet sur lequel appliquer l'action
 * @param  int    $id    Identifiant de l'objet
 * @param  array  $qui   Description de l'auteur demandant l'autorisation
 * @param  array  $opt   Options de cette autorisation
 * @return bool          true s'il a le droit, false sinon
**/
function autoriser_campagne_associer_dist($faire, $type, $id, $qui, $opt) {
	$ca = sql_fetsel('*','spip_biodiv_campagnes','id_campagne='.intval($id));
	if(is_array($ca)) {
		$d = date_parse_from_format("Y-m-d H:i:s",$ca['debut']);
		$f = date_parse_from_format("Y-m-d H:i:s",$ca['fin']);
		$debut = mktime($d['hour'],$d['minute'],$d['second'],$d['month'],$d['day'],$d['year']);
		$fin = mktime($f['hour'],$f['minute'],$f['second'],$f['month'],$f['day'],$f['year']);
		$now = time();
		return(($now >= $debut) and ($now <= $fin));
	}
	return false;
}