<?php
/**
 * Fonctions pour Sibmail
 *
 * @plugin     Sibmail
 * @copyright  2023
 * @author     Renaud LAURETTE
 * @licence    GNU/GPL
 * @package    SPIP\Sibmail\Fonctions
 */


/**
 * Fonction générale de transmission de Requête à Brevo via l'API
 * La requête est un tableau dont les entrées sont
 *  - api : la partie spécifique de l'URL pour l'API, y compris la query string
 *  - cmd : le type de requete : GET, POST, PUT, ...
 *  - data : un texte JSON contenant le corps de la requête
 * La clé API est prise dans la configuration du plug-in.
 *
 * si $jsonReturn=true, la fonction renvoie un objet contenant 2 champs
 *	code = le code de retour HTTP de la requete
 *  data = un objet représentant le décodage de la réponse json du serveur
 * Si $jsonReturn=false (par défaut), la fonction renvoie seulement le
 *  contenu textuel retourné par le serveur
 * 
 * @param array $request - la requête
 * @param boolean $jsonReturn 
 * @return string - une chaine JSON contenant la réponse du serveur Brevo
 */
 
function sibmail_sendRequest($request,$jsonReturn=false) {
	$curl = curl_init();
	curl_setopt_array($curl, array(
		CURLOPT_URL => 'https://api.sendinblue.com/v3/'.$request['api'],
		CURLOPT_RETURNTRANSFER => true,
		CURLOPT_ENCODING => '',
		CURLOPT_MAXREDIRS => 10,
		CURLOPT_TIMEOUT => 0,
		CURLOPT_FOLLOWLOCATION => true,
		CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
		CURLOPT_CUSTOMREQUEST => $request['cmd'],
		CURLOPT_POSTFIELDS => $request['data'],
		CURLOPT_HTTPHEADER => array(
			'api-key: ' . lire_config('sibmail/apikey'),
			'Content-Type: application/json'
			),
		));

	$response = curl_exec($curl);
	if($jsonReturn) {
		$answer = new stdClass();
		$answer->code = curl_getinfo($curl,CURLINFO_HTTP_CODE);
		$answer->data = json_decode($response, false); // décodage en objet
		return $answer;
	}	
	curl_close($curl);
	return $response;
}

/**
 * Fonction d'envoi d'un événement à Brevo
 *
 * @param string $email			Adresse mail de la personne ayant déclenché l'événement
 * @param string $event			Nom de l'événement
 * @param string $id			identifiant de l'événement (obligatoirement une string)
 * @param object|null $props	Un objet portant des attributs de contact à mettre à jour (optionnel)
 * @param object|null $data		Un objet portant les détails de l'événement (optionnel)
 *
*/
function sibmail_sendEvent($email,$event,$id,$props,$data) {
	$actif = lire_config('sibmail/activer_transactionnel');
	if($actif == 'on') {
		spip_log("SIBMAIL::SendEvent($email,$event,$id)","sibmail."._LOG_INFO_IMPORTANTE);
	
		$request = new stdClass();
		$request->email = $email;
		$request->event = $event;
	
		$eventData = new stdClass();
		$eventData->id = (string)$id;
		if(is_object($data)) {
			$eventData->data = $data;
		}
		$request->eventdata = $eventData;
		
		if(is_object($props)){
			$request->properties = $props;
		}
	
		$curl = curl_init();
		$txtrq = json_encode($request);
		curl_setopt_array($curl, array(
			CURLOPT_URL => 'https://in-automate.brevo.com/api/v2/trackEvent',
			CURLOPT_RETURNTRANSFER => true,
			CURLOPT_ENCODING => '',
			CURLOPT_MAXREDIRS => 10,
			CURLOPT_TIMEOUT => 0,
			CURLOPT_FOLLOWLOCATION => true,
			CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
			CURLOPT_CUSTOMREQUEST => 'POST',
			CURLOPT_POSTFIELDS => $txtrq,
			CURLOPT_HTTPHEADER => array(
				'ma-key: ' . lire_config('sibmail/makey'),
				'Content-Type: application/json'
				),
			));

		spip_log("SIBMAIL::SendEvent - Request: $txtrq","sibmail."._LOG_INFO_IMPORTANTE);
		$response = curl_exec($curl);
		$code = (curl_getinfo($curl,CURLINFO_HTTP_CODE) == 204) ? true : false;
		curl_close($curl);
		return $code;
	}
	return true;
}


/**
 * Enregistrement d'un auteur en tant que contact et inscription aux listes de distribution
 * La configuration du plugin prévoit une liste pour tous les inscrits sur le site (admin, rédac, visiteur ou externe)
 * Le statut est identifié sur l'attribut 'attribut_statut' de la configuration du plugin. 
 *
 * @param integer $id_auteur - Identifiant de l'auteur dans la base spip
 * @param array|null $options - options
 */

function sibmail_subscribe($id_auteur, $options=null) {
	$auteur = sql_fetsel('*','spip_auteurs',"id_auteur=$id_auteur");
	
	/* Pas d'auteur, pas d'inscription */
	if(!$auteur) {
			spip_log("SIBMAIL::Subscribe - auteur $id_auteur inconnu","sibmail."._LOG_INFO_IMPORTANTE);
			return;
	}

	/* association à une liste */
	$listes = array(
			intval(lire_config('sibmail/liste_inscrits')),
			);
	
	/* une adresse mail ? */
	if(!strpos($auteur['email'],'@')) {
		spip_log("SIBMAIL::Subscribe - auteur $id_auteur : email invalide: ".$auteur['email'],"sibmail."._LOG_INFO_IMPORTANTE);
		return;
	}
	
	/* Recherche de l'attribut Brevo dans lequel stocker le pseudo et le statut de l'auteur */
	$pseudo = strtoupper(lire_config('sibmail/attribut_pseudo'));
	$statutSite = strtoupper(lire_config('sibmail/attribut_statut'));
	
	/* On a un auteur valide avec un email */	
	$data = new stdClass();
	$data->email = $auteur['email'];
	$attribs = new stdClass();
	$attribs->$pseudo = $auteur['nom'];
	$attribs->$statutSite = _T('sibmail:'.$auteur['statut']);
	if(is_array($options)) {
		if(isset($options['nom'])) $attribs->NOM = $options['nom'];
		if(isset($options['prenom'])) $attribs->PRENOM = $options['prenom'];
		if(isset($options['listes'])) $listes = array_merge($listes,$options['to']);
	}
	$data->attributes = $attribs;
	$data->listIds = $listes;
	$data->updateEnabled = true;
	$request['data'] = json_encode($data);	
	
	if(isset($auteur['extra'])) {
		$extra = @unserialize($auteur['extra']);
	} else $extra = null;
		
	// Envoi de la requête de souscription
	$request['cmd'] = 'POST';
	$request['api'] = "contacts";
	spip_log("SIBMAIL::Sending request ".serialize($request),"sibmail."._LOG_INFO_IMPORTANTE);
	$response = json_decode(sibmail_sendRequest($request));
	spip_log("SIBMAIL::Received answer ".serialize($response),"sibmail."._LOG_INFO_IMPORTANTE);
		
	// On interroge Brevo pour vérifier que les données utilisateur ont été mises à jour
	// et on les stocke dans le champ extra de l'auteur
	$getuser = array(
		'cmd' => 'GET',
		'api' => 'contacts/' . $auteur['email'],
		'data' => ''
		);
	$controle = json_decode(sibmail_sendRequest($getuser));
	if($controle->email == $auteur['email']) {
		$extra['sibmail'] = array(
			'id' => $controle->id,
			'createdAt' => $controle->createdAt,
			'modifiedAt' => $controle->modifiedAt,
			'listIds' => $controle->listIds
			);
		spip_log("SIBMAIL::Author $id_auteur info : ".serialize($extra['sibmail']),"sibmail."._LOG_INFO_IMPORTANTE);
		sql_updateq('spip_auteurs',array('extra'=>serialize($extra)),'id_auteur='.intval($id_auteur));
	} else {
		spip_log("SIBMAIL::Subscription failed for author $id_auteur : unknown email ".$auteur['email'],"sibmail."._LOG_INFO_IMPORTANTE);
	}
	return;
}

/**
 * Création d'une campagne de mailing
 *
 * @param string $titre 	- le titre de la campagne dans la base Sendinblue/Brevo
 * @param string $url 		- l'URL de la page qui constituera le message de la campagne
 * @param array|integer $listes 	- un tableau de numéros de listes de distribution ou un numéro unique
 * @param string $sujet		- le sujet du mail qui sera envoyé
 * @param integer $emetteur	- l'identifiant Brevo de l'emetteur
 * @return integer|null		- le numéro de la campagne créée ou bien null
 *
 * Crée une campagne de mailing avec les paramètre considérés.
 * L'expéditeur du message aura pour email celui du webmestre. 
 * Cette adresse mail doit être déclarée comme émetteur reconnu dans sendinblue
 *
 * Note : la fonction crée la campagne mais ne la déclenche pas.
 * Pour déclencher la campagne, utiliser ensuite sibmail_activateCampaign
 *
 */
function sibmail_createCampaign($titre,$emetteur,$listes,$sujet,$url) {
	
	spip_log("SIBMAIL::Création campagne de $emetteur pour ".serialize($listes),"sibmail."._LOG_INFO_IMPORTANTE);
	$sender = new stdClass();
	if($emetteur) {	
		/* valeur nom nulle : on cherche l'emeteur dans la base Brevo */
		$emetteur = intval($emetteur);
		$allSenders = sibmail_getAllSenders();
		$found = false;
		foreach($allSenders as $oneSender) {
			if($oneSender['id'] == $emetteur) {
				$found = true;
				$sender->name = $oneSender['name'];
				$sender->email = $oneSender['email'];
			}
		}
		if (!$found) {
			/* pas d'emetteur trouvé : on arrête */
			return null;
		}
	} else {
		/* 	Pas d'emetteur désigné, on tente le webmestre. 
			Attention : si cet emetteur n'est pas déclaré dans Brevo comme 'sender'
			la création de la campagne de mailing échouera !
		*/
		$sender->name = lire_config('nom_site');
		$sender->email = lire_config('email_webmestre');
	}
	
	$recipient = new stdClass();
	$recipient->listIds = is_array($listes) ? $listes : array($listes);
	
	$data = new stdClass();
	$data->inlineImageActivation = false;
	$data->sendAtBestTime = false;
	$data->abTesting = false;
	$data->ipWarmupEnable = false;
	$data->name = $titre;
	$data->htmlUrl = $url;
	$data->toField = '{{NOM}}';
	$data->footer = '[DEFAULT_FOOTER]';
	$data->subject = $sujet;
	$data->recipients = $recipient;
	$data->sender = $sender;
	$data->replyTo = $sender->email;
	
	$request=array();
	$request['cmd'] = 'POST';
	$request['api'] = 'emailCampaigns';
	$request['data'] = json_encode($data);
	
	spip_log("SIBMAIL::Création campagne : ".$request['data'],"sibmail."._LOG_INFO_IMPORTANTE);
	
	$response = json_decode(sibmail_sendRequest($request));
	if(is_object($response) and isset($response->id)) {
		/* création réussie */
		spip_log("SIBMAIL::Création campagne réussie : ".serialize($response),"sibmail."._LOG_INFO_IMPORTANTE);
		return $response->id;
	}
	
	/* échec */
	spip_log("SIBMAIL::Création campagne ECHEC : ".serialize($response),"sibmail."._LOG_INFO_IMPORTANTE);
	return null;
}

/**
 * Activation de la campagne de mailing (envoi des mails)
 *
 * @param integer $cid - identifiant de la campagne
 * @return integer - 1 si le lancement est un succès
 *
 */
function sibmail_activateCampaign($cid) {
	if($cid) {
		$request = array(
			'cmd' => 'POST',
			'api' => "emailCampaigns/$cid/sendNow",
			'data' => ""
			);		
		
		$response = sibmail_sendRequest($request);
		spip_log("SIBMAIL::Activation campagne $cid : '$response'","sibmail."._LOG_INFO_IMPORTANTE);
		return true;
	}
	return false;
}

/**
  * Rapport de campagne
  */
  
function sibmail_getCampaignReport($cid) {
	$data = array();
	spip_log("SIBMAIL::Demande rapport campagne $cid ","sibmail."._LOG_INFO_IMPORTANTE);
	if($cid and autoriser('consulter','mailing')) {
		$request = array(
			'cmd' => 'GET',
			'api' => "emailCampaigns/$cid",
			'data' => ""
			);				
		
		$answer = sibmail_sendRequest($request);
		$response = json_decode($answer);
		spip_log("SIBMAIL::Rapport campagne $cid : '$answer'","sibmail."._LOG_INFO_IMPORTANTE);
		if(is_object($response) and isset($response->id) and ($response->id == $cid)) {	
			$data['id'] = $response->id;
			$data['name'] = $response->name;
			$data['status'] = $response->status;
			$data['senderName'] = $response->sender->name;
			$data['senderEmail'] = $response->sender->email;
			$data['replyTo'] = $response->replyTo;
			$data['previewText'] = $response->previewText;
			$data['tag'] = $response->tag;
			$data['subject'] = $response->subject;
			$data['scheduledAt'] = $response->scheduledAt;
			$data['createdAt'] = $response->createdAt;
			$data['modifiedAt'] = $response->modifiedAt;
			$data['sentDate'] = $response->sentDate;
			$data['sharelink'] = $response->shareLink;
			
			foreach($response->statistics->campaignStats as $cmp)
				$data['stats'][] = (array)$cmp;
			foreach($response->statistics->linksStats as $key => $value) {
				$data['links'][]=$key;	
			}
		}
	}
	return $data;
}

/**
  * Envoie par email les informations de profil à son propriétaire
  * L'emetteur du message est le webmestre du site, qui doit avoir été déclaré comme expéditeur Brévo
  * Le contenu du message est défini par des squelettes :
  *  - infos_profil_html.html pour le message en HTML
  *  - infos_profil_txt.html pour le message en texte brut (UTF-8)
  *
  * @param array $user	- les informations de profil
  */
function sibmail_transmitContactData($user) {
	
	/* préparation de l'emetteur et du destinataire */
	$data = new stdClass();
	$data->sender->name = 'Notifications ' . lire_config('nom_site_spip');
	$data->sender->email = lire_config('email_webmestre');
	$data->to[]->email = $user['email'];
	$data->subject = _T('sibmail:infos_profil');
	
	/* contenu HTML */
	$htmlContent = recuperer_fond('infos_profil_html',$user);
	
	/* contenu texte alternatif */
	$textContent = recuperer_fond('infos_profil_txt',$user);
	
	/* finalisation des données */
	$data->htmlContent = $htmlContent;
	$data->textContent = $textContent;
	
	/* requete */
	$requete = array(
		'cmd' => 'POST',
		'api' => 'smtp/email',
		'data' => json_encode($data)
		);
	$response = json_decode(sibmail_sendRequest($request));
}

/**
 * Filtre retournant un tableau avec les informations de contact
 * Renvoie un tableau vide aux personnes non habilitées, mais adresse alors
 * un email au contact avec les informations.
 *
 * @param String email - l'adresse mail du contact recherché
 * @return Array
 * 
 */
function getContactDetails($email) {
	
	$request = array(
		'cmd' => 'GET',
		'api' => "contacts/$email",
		'data' => ""
		);
	$response = json_decode(sibmail_sendRequest($request));
	$pseudo = strtoupper(lire_config('sibmail/attribut_pseudo'));
	
	if(is_object($response) and isset($response->email) and ($response->email == $email)) {
		// Le contact existe : on récupère ses données ...
		$user = array(
					'email' => $response->email,
					'id' => $response->id,
					'creation' => $response->createdAt,
					'modification' => $response->modifiedAt,
					'nom' => $response->attributes->NOM,
					'prenom' => $response->attributes->PRENOM,
					'pseudo' => $response->attributes->$pseudo,
					'listes' => array()
					);
		// ... et les listes auxquelles il appartient
		if( isset($response->listIds) and is_array($response->listIds) and count($response->listIds) ) {
			$lists = sibmail_getAllLists();
			foreach($response->listIds as $lid) {
				$lid = intval($lid);
				if(array_key_exists($lid, $lists)) {
					$user['listes'][] = array(
						'id' => $lid,
						'nom'=> $lists[$lid]['name']
						);
				}
			}
		}
		// On a les informations. Peut-on les communiquer ?
		
		// L'adresse mail correspond-elle à un auteur ?
		$auteur = sql_fetsel('*','spip_auteurs',"email=".sql_quote($email));
		if($auteur) {
			// C'est un auteur - a-t-on le droit de communiquer ses données ?
			if(autoriser('modifier','auteur',$auteur['id_auteur'])) {
				// oui - on retourne les données
				return $user;
			} else {
				// non - on demande à Brevo d'envoyer les infos par mail à l'auteur
				// On ne retourne rien de visible
				sibmail_transmitContactData($user);
				return array();
			}
		} else { 
			// ce n'est pas une personne connue sur le site
			if(autoriser('webmestre')) {
				// le webmestre a tous les droits ...
				return $user;
			} else {
				// Mais sinon, aucun moyen de contrôler les droits, alors
				// on demande à Brevo d'envoyer les infos par mail au contact concerné
				// On ne retourne rien de visible
				sibmail_transmitContactData($user);
				return array();
			}
		}	
	} else {
		// adresse mail inconnue
		// Pas de message d'erreur : ce serait déjà une information ...
		return array();
	}		
}

/**
 * Retourne les 50 premières listes si le paramètre d'appel est null ou absent
 * Retourne les listes du folder dont l'id est fourni sinon
 *
 * @param integer $folderId	- identifiant du folder (optionnel)
 * @return array			- un tableeau contenant les listes recherchées
 *
 */
function sibmail_getAllLists($folderId=null) {
	if($folderId) {
		$folderId = intval($folderId);		
		$api = "contacts/folders/$folderId/lists";
	} else {
		$api = 'contacts/lists?limit=50&offset=0&sort=desc';
	}
	
	$request = array(
		'cmd' => 'GET',
		'api' => $api,
		'data' => ""
		);
	$lists = array();
	$response = json_decode(sibmail_sendRequest($request));
	if(is_object($response) and isset($response->lists)) {
		foreach($response->lists as $list) {
			$lists[intval($list->id)] = array(
				'id' => $list->id,
				'name' => $list->name,
				'members' => $list->uniqueSubscribers
				);
		}
	}
	return $lists;
}

/**
 * Retourne la liste des expéditeurs valides dans Brevo
 *
 * @return array	- un tableau de tableaux
 */
function sibmail_getAllSenders() {
	$request = array(
		'cmd' => 'GET',
		'api' => 'senders',
		'data' => ""
		);
	$mySenders = array();
	$response = json_decode(sibmail_sendRequest($request));
	if(is_object($response) and isset($response->senders)) {
		foreach($response->senders as $sender) {
			if($sender->active == true) {
				$mySenders[intval($sender->id)] = array(
					'id' => $sender->id,
					'name' => $sender->name,
					'email' => $sender->email
					);
			}
		}
	}
	return $mySenders;
}

/**
  * Filtre pour formulaire
  * Retourne un tableau dont chaque élément est un objet demandé (lui-même un tableau)
  *
  * #VAL{liste}|sibListe				- toutes les listes
  * #VAL{liste}|sibListe{abonnement}	- toutes les listes auxquelles un visiteur peut s'abonner
  * #VAL{emetteur}|sibListe				- tous les emetteurs
  *
  * Les listes proposées à l'abonnement sont celles qui se trouvent dans le folder correspondant
  * tel que défini dans la configuration du plugin.
  *
  * @param string $what	- Nature des informations demandées
  * @param string $option	- 'abonnement' (optionnel)
  */
function sibListe($what,$option=null) {
	switch($what) {
	case 'liste':
		$folderId = null;
		if($option == 'abonnement') {
			$folderId = lire_config('sibmail/dossier_abonnement');
		}	
		// spip_log("SIBMAIL::Recherche listes (Option $option folder $folderId)","sibmail."._LOG_INFO_IMPORTANTE);
		return sibmail_getAllLists($folderId);
		break;
	case 'emetteur':
		return sibmail_getAllSenders();
		break;
	default:
		return array();
		break;
	}
}



